% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/joining.R
\name{joining}
\alias{joining}
\alias{join_}
\alias{left_join_}
\alias{right_join_}
\alias{inner_join_}
\alias{full_join_}
\alias{semi_join_}
\alias{anti_join_}
\title{Joining Functions}
\usage{
join_(
  x,
  y,
  by = NULL,
  copy = FALSE,
  suffix = c(".x", ".y"),
  ...,
  keep = NULL,
  na_matches = c("na", "never"),
  multiple = "all",
  unmatched = "drop",
  relationship = NULL,
  sort = FALSE,
  verbose = 0,
  column = NULL,
  attr = NULL,
  how = "full"
)

left_join_(
  x = (.),
  y,
  by = NULL,
  copy = FALSE,
  suffix = c(".x", ".y"),
  ...,
  keep = NULL,
  na_matches = c("na", "never"),
  multiple = "all",
  unmatched = "drop",
  relationship = NULL,
  sort = FALSE,
  verbose = 0,
  column = NULL,
  attr = NULL
)

right_join_(
  x = (.),
  y,
  by = NULL,
  copy = FALSE,
  suffix = c(".x", ".y"),
  ...,
  keep = NULL,
  na_matches = c("na", "never"),
  multiple = "all",
  unmatched = "drop",
  relationship = NULL,
  sort = FALSE,
  verbose = 0,
  column = NULL,
  attr = NULL
)

inner_join_(
  x = (.),
  y,
  by = NULL,
  copy = FALSE,
  suffix = c(".x", ".y"),
  ...,
  keep = NULL,
  na_matches = c("na", "never"),
  multiple = "all",
  unmatched = "drop",
  relationship = NULL,
  sort = FALSE,
  verbose = 0,
  column = NULL,
  attr = NULL
)

full_join_(
  x = (.),
  y,
  by = NULL,
  copy = FALSE,
  suffix = c(".x", ".y"),
  ...,
  keep = NULL,
  na_matches = c("na", "never"),
  multiple = "all",
  relationship = NULL,
  sort = FALSE,
  verbose = 0,
  column = NULL,
  attr = NULL
)

semi_join_(
  x = (.),
  y,
  by = NULL,
  copy = FALSE,
  ...,
  na_matches = c("na", "never"),
  sort = FALSE,
  verbose = 0,
  column = NULL,
  attr = NULL
)

anti_join_(
  x = (.),
  y,
  by = NULL,
  copy = FALSE,
  ...,
  na_matches = c("na", "never"),
  sort = FALSE,
  verbose = 0,
  column = NULL,
  attr = NULL
)
}
\arguments{
\item{x}{A data frame (data.frame, data.table, or tibble)}

\item{y}{A data frame to join with x}

\item{by}{A character vector of column names to join by. If \code{NULL}, uses all
columns with common names across x and y. Can also be a named character
vector to join by different column names (e.g., \code{c("a" = "b")} joins x$a to
y$b), or a \code{dplyr::join_by()} object for complex joins.}

\item{copy}{If x and y are not from the same data source, and copy is \code{TRUE},
then y will be copied into the same source as x. This allows you to join
tables across data sources, but is potentially expensive.}

\item{suffix}{Character vector of length 2 specifying suffixes to append to
duplicate column names. Default is \code{c(".x", ".y")}.}

\item{...}{Additional arguments (currently unused, for compatibility)}

\item{keep}{Should the join keys from both x and y be preserved in the output?
If \code{NULL} (default), keeps join keys from x only for equality joins, but
keeps keys from both for inequality joins. If \code{TRUE}, all keys from both
inputs are retained. If \code{FALSE}, only keys from x are retained. Note: when
\code{keep = TRUE}, calculation is delegated to dplyr join methods.}

\item{na_matches}{Should two \code{NA} or two \code{NaN} values match? \code{"na"} (default)
treats two \code{NA} or two \code{NaN} values as equal, like \code{\%in\%}, \code{match()}, and
\code{merge()}. \code{"never"} treats two \code{NA} or two \code{NaN} values as different, and
will never match them together or to any other values. When \code{"never"},
calculation is delegated to dplyr join methods.}

\item{multiple}{Handling of rows in x with multiple matches in y. Options:
\code{"all"} (default) returns every match detected in y (SQL behavior),
\code{"first"} returns the first match detected in y,
\code{"last"} returns the last match detected in y,
\code{"any"} returns one match (faster but non-deterministic).
For \code{"any"} and \code{"last"} (and \code{"first"} for right joins), calculation is
delegated to dplyr join methods.}

\item{unmatched}{How should unmatched keys that would result in dropped rows
be handled? \code{"drop"} (default) drops unmatched keys from the result.
\code{"error"} throws an error if unmatched keys are detected. Can also be a
named list like \code{list(x = 1, y = 0.5, fail = "warning")} specifying the
proportions that must match and the action (\code{"message"}, \code{"warning"}, or
\code{"error"}). Not available for \code{full_join_()}, \code{semi_join_()}, and
\code{anti_join_()}.}

\item{relationship}{Expected relationship between the keys of x and y:
\code{NULL} (default) has no expectations but warns for many-to-many,
\code{"one-to-one"} expects each row in x matches at most 1 row in y and vice versa,
\code{"one-to-many"} expects each row in y matches at most 1 row in x,
\code{"many-to-one"} expects each row in x matches at most 1 row in y,
\code{"many-to-many"} has no restrictions (explicit about relationship).}

\item{sort}{Logical. If \code{TRUE}, the result is sorted by the grouping variables.
Default is \code{FALSE}.}

\item{verbose}{Integer controlling information printed about the join:
\code{0} means no output (default), \code{1} prints join information, and
\code{2} additionally prints the classes of the \code{by} columns.
Note: This parameter is ignored when using dplyr join methods.}

\item{column}{Name for an extra column to generate in the output indicating
which dataset a record came from. \code{TRUE} creates a column named \code{".join"},
or provide a custom name as a character string. \code{NULL} (default) does not
add this column. Note: This parameter is ignored when using dplyr join methods.}

\item{attr}{Name for an attribute providing information about the join
performed (including output of \code{\link[collapse:fmatch]{collapse::fmatch()}}) attached to the result.
\code{TRUE} creates an attribute named \code{"join.match"}, or provide a custom name.
\code{NULL} (default) does not add this attribute. Note: This parameter is
ignored when using dplyr join methods.}

\item{how}{Character string specifying the join type for \code{join_()}:
\code{"full"} (default), \code{"inner"}, \code{"left"}, \code{"right"}, \code{"semi"}, or \code{"anti"}.}
}
\value{
A data frame of the same type as \verb{x``. The order of the rows and columns of }x` is preserved as much as possible.
\itemize{
\item \code{left_join_()} returns all rows from \code{x}, and all columns from \code{x} and \code{y}
\item \code{right_join_()} returns all rows from \code{y}, and all columns from \code{x} and \code{y}
\item \code{inner_join_()} returns only rows with matches in both \code{x} and \code{y}
\item \code{full_join_()} returns all rows from both \code{x} and \code{y}
\item \code{semi_join_()} returns rows from \code{x} (no columns added from \code{y})
\item \code{anti_join_()} returns rows from \code{x} with no match in \code{y} (no columns from
\code{y})
}
}
\description{
Functions for joining two data frames based on matching values in key columns.

These are SciViews::R versions of tidyverse functions with standard
evaluation and formula-based non-standard evaluation (ending with underscore
\verb{_}). They work with data.frame, data.table, and tibbles.

\strong{Functions:}
\itemize{
\item \code{left_join_()} - Keep all rows from x, add matching columns from y
\item \code{right_join_()} - Keep all rows from y, add matching columns from x
\item \code{inner_join_()} - Keep only rows with matches in both x and y
\item \code{full_join_()} - Keep all rows from both x and y
\item \code{semi_join_()} - Keep rows in x that have a match in y (no columns from y)
\item \code{anti_join_()} - Keep rows in x that do NOT have a match in y
\item \code{join_()} - Generic join function with \code{how} parameter
}
}
\examples{
library(svTidy)

# Create example datasets
band_members <- data.frame(
  name = c("Mick", "John", "Paul"),
  band = c("Stones", "Beatles", "Beatles")
)
band_instruments <- data.frame(
  name = c("John", "Paul", "Keith"),
  plays = c("guitar", "bass", "guitar")
)

# Left join - keep all rows from x
band_members |> left_join_(band_instruments, by = "name")

# Right join - keep all rows from y
band_members |> right_join_(band_instruments, by = "name")

# Inner join - keep only matching rows
band_members |> inner_join_(band_instruments, by = "name")

# Full join - keep all rows from both
band_members |> full_join_(band_instruments, by = "name")

# Semi join - filter x to rows matching y (no columns from y)
band_members |> semi_join_(band_instruments, by = "name")

# Anti join - filter x to rows NOT matching y
band_members |> anti_join_(band_instruments, by = "name")

# Join by different column names
band_instruments2 <- data.frame(
  artist = c("John", "Paul", "Keith"),
  plays = c("guitar", "bass", "guitar")
)
band_members |> left_join_(band_instruments2, by = c("name" = "artist"))

# Add suffix to duplicate columns
df1 <- data.frame(id = 1:3, value = c("a", "b", "c"))
df2 <- data.frame(id = 2:4, value = c("B", "C", "D"))
df1 |> full_join_(df2, by = "id", suffix = c("_x", "_y"))

# Control handling of multiple matches
df_x <- data.frame(key = c(1, 1, 2), x = c("a", "b", "c"))
df_y <- data.frame(key = c(1, 1, 2), y = c("A", "B", "C"))
df_x |> left_join_(df_y, by = "key", multiple = "all")
df_x |> left_join_(df_y, by = "key", multiple = "first")

# Validate relationships
df_one <- data.frame(id = 1:3, val = c("a", "b", "c"))
df_many <- data.frame(id = c(1, 1, 2), val = c("A", "B", "C"))
\dontrun{
# This will error - expects one-to-one but is one-to-many
df_one |> left_join_(df_many, by = "id", relationship = "one-to-one")
}
# This works - explicitly one-to-many
df_one |> left_join_(df_many, by = "id", relationship = "one-to-many")

# Add indicator column showing source
band_members |>
  full_join_(band_instruments, by = "name", column = "source")

# Use generic join_() with how parameter
band_members |> join_(band_instruments, by = "name", how = "inner")
band_members |> join_(band_instruments, by = "name", how = "left")

# Handle unmatched keys
\dontrun{
# Error if any keys don't match
band_members |>
  inner_join_(band_instruments, by = "name", unmatched = "error")
}
}
\seealso{
\code{\link[dplyr:mutate-joins]{dplyr::left_join()}}, \code{\link[dplyr:mutate-joins]{dplyr::right_join()}}, \code{\link[dplyr:mutate-joins]{dplyr::inner_join()}},
\code{\link[dplyr:mutate-joins]{dplyr::full_join()}}, \code{\link[dplyr:filter-joins]{dplyr::semi_join()}}, \code{\link[dplyr:filter-joins]{dplyr::anti_join()}},
\code{\link[collapse:join]{collapse::join()}}, \code{\link[dplyr:join_by]{dplyr::join_by()}}
}
