% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mutating.R
\name{mutating}
\alias{mutating}
\alias{mutate_}
\alias{transmute_}
\title{Mutating Functions}
\usage{
mutate_(
  .data = (.),
  ...,
  .by = NULL,
  .keep = "all",
  .before = NULL,
  .after = NULL,
  .cols = NULL
)

transmute_(.data, ...)
}
\arguments{
\item{.data}{A data frame (data.frame, data.table, or tibble)}

\item{...}{Name-value pairs specifying new columns or modifications. Names
are column names; values are expressions to compute. Use formulas for
non-standard evaluation (e.g., \code{new_col = ~mean(old_col)}), or provide
character names and expressions for standard evaluation. New columns can
refer to columns just created. Within formulas, you can provide column
names use like this: \code{'new_col_name' ~ median(old_col)}. You can also use
the name of an object that contains the column name. For instance,
\code{col_names <- c("new_col1", "new_col2")}, then in \code{mutate_()}, use
\code{col_name[1] ~ median(old_col)}.}

\item{.by}{Optional grouping variables for per-group computations. Provide
as formulas (e.g., \code{~group_col}) or character names. Groups are temporary
and not preserved in the output. Cannot be used with grouped data frames.}

\item{.keep}{Control which columns to keep in the output:
\itemize{
\item \code{"all"} (default) - Keep all existing columns plus new ones
\item \code{"used"} - Keep columns used to make new columns, and new columns
\item \code{"unused"} - Keep columns not used to make new columns, and new columns.
This is useful if you do not need the columns that are used to create the
new ones.
\item \code{"none"} - Keep only new columns (same as \code{transmute_()})
}}

\item{.before}{Optional column name or position to place new columns before.
Not yet implemented - use dplyr::mutate() instead.}

\item{.after}{Optional column name or position to place new columns after.
Not yet implemented - use dplyr::mutate() instead.}

\item{.cols}{Optional character vector of column names to operate on. If
provided, only these columns are modified or created.}
}
\value{
A data frame of the same type as \code{.data} with modified or new columns.
\itemize{
\item \code{mutate_()} returns all columns (by default), including new/modified ones
\item \code{transmute_()} returns only the newly created columns
}
}
\description{
Functions for creating new columns or modifying existing columns in a data
frame. These are SciViews::R versions of tidyverse functions with standard
evaluation and formula-based non-standard evaluation (ending with underscore
\verb{_}). They work with data.frame, data.table, and tibbles.

\strong{Functions:}
\itemize{
\item \code{mutate_()} - Add new columns or modify existing ones
\item \code{transmute_()} - Create new columns and drop all others
}
}
\examples{
library(svTidy)
data(mtcars)

# Create new columns using formulas
mtcars |> mutate_(hp_per_cyl = ~hp/cyl)

# Multiple new columns
mtcars |>
  mutate_(
    hp_per_cyl = ~hp/cyl,
    wt_kg = ~wt * 453.592
  )

# Modify existing column
mtcars |> mutate_(mpg = ~mpg * 1.5)

# Reference newly created columns
mtcars |>
  mutate_(
    hp_per_cyl = ~hp/cyl,
    hp_per_cyl_scaled = ~hp_per_cyl * 100
  )

# Use column name in a variable
col_name <- "power_ratio"
mtcars |> mutate_(col_name ~ hp/wt)

# Group-wise computations with .by
mtcars |>
  mutate_(
    mpg_centered = ~mpg - mean(mpg),
    .by = 'cyl'
  )

# Multiple grouping variables
mtcars |>
  mutate_(
    hp_rank = ~rank(hp),
    .by = c('cyl', 'gear')
  )

# Control which columns to keep
mtcars |>
  mutate_(
    hp_per_cyl = ~hp/cyl,
    .keep = "used"
  )

mtcars |>
  mutate_(
    efficiency = ~mpg/hp,
    .keep = "unused"
  )

# transmute_() keeps only new columns
mtcars |>
  transmute_(
    car = ~rownames(mtcars),
    hp_per_cyl = ~hp/cyl,
    efficiency = ~mpg/wt
  )

# Conditional mutations
mtcars |>
  mutate_(
    performance = ~ifelse(hp > 150, "high", "normal")
  )

# Use with grouped data
mtcars |>
  group_by_(~cyl) |>
  mutate_(mpg_ratio = ~mpg/mean(mpg))

# Complex transformations
mtcars |>
  mutate_(
    log_hp = ~log(hp),
    sqrt_wt = ~sqrt(wt),
    hp_wt_interaction = ~hp * wt
  )

}
\seealso{
\code{\link[dplyr:mutate]{dplyr::mutate()}}, \code{\link[dplyr:transmute]{dplyr::transmute()}}, \code{\link[collapse:ftransform]{collapse::fmutate()}}
}
