% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pivoting.R
\name{pivoting}
\alias{pivoting}
\alias{pivot_longer_}
\alias{pivot_wider_}
\title{Pivoting Functions}
\usage{
pivot_longer_(
  .data = (.),
  cols,
  ...,
  cols_vary = "fastest",
  names_to = "name",
  names_prefix = NULL,
  values_to = "value",
  values_drop_na = FALSE,
  factor = FALSE
)

pivot_wider_(
  .data = (.),
  ...,
  id_cols = NULL,
  id_expand = FALSE,
  names_from = "",
  names_prefix = "",
  names_vary = "fastest",
  values_from = "",
  values_fill = NULL,
  values_fn = "last",
  drop = TRUE,
  sort = FALSE
)
}
\arguments{
\item{.data}{A data frame (data.frame, data.table, or tibble)}

\item{cols}{For \code{pivot_longer_()}: columns to pivot into longer format. Use
tidy-select syntax with formulas (e.g., \code{~col1:col5}, \code{~starts_with("x")}),
or provide column positions or names as a vector.}

\item{...}{Additional arguments (currently unused, for compatibility)}

\item{cols_vary}{Character. Either \code{"fastest"} (default) or \code{"slowest"}. If
\code{"fastest"}, keeps individual rows from \code{cols} close together in the output.
If \code{"slowest"}, keeps individual columns from \code{cols} close together.}

\item{names_to}{Character string specifying the name of the column to create
from the column names being pivoted. Default is \code{"name"}.}

\item{names_prefix}{Character. A regular expression used to remove matching
text from the start of each variable name before creating the names column.}

\item{values_to}{Character string specifying the name of the column to create
from the cell values. Default is \code{"value"}.}

\item{values_drop_na}{Logical. If \code{TRUE}, rows containing only \code{NA} values in
the \code{values_to} column are dropped from the result. Default is \code{FALSE}.}

\item{factor}{Logical. If \code{TRUE}, convert the names and values columns to
factors. If \code{FALSE} (default), leave them as character strings.}

\item{id_cols}{For \code{pivot_wider_()}: columns that uniquely identify each
observation. Use tidy-select syntax or \code{NULL} (default) to use all columns
except \code{names_from} and \code{values_from}.}

\item{id_expand}{Logical. If \code{TRUE}, expand the \code{id_cols} to include all
possible combinations. Default is \code{FALSE}.}

\item{names_from}{For \code{pivot_wider_()}: column(s) containing the names for
the new columns. Provide as unquoted names or character vector. Default is
\code{name}.}

\item{names_vary}{Character. How column names are constructed when multiple
\code{names_from} or \code{values_from} columns are provided: \code{"fastest"} (default),
\code{"slowest"}, \code{"transpose"}, or \code{"slowtranspose"}.}

\item{values_from}{For \code{pivot_wider_()}: column(s) containing the values for
the new columns. Provide as unquoted names or character vector. Default is
\code{value}.}

\item{values_fill}{Optional scalar value to use for missing combinations.
Default is \code{NULL} (use \code{NA}).}

\item{values_fn}{Function to apply when there are multiple values for a cell.
Can be a string naming an internal function (\code{"first"}, \code{"last"} (default),
\code{"count"}, \code{"sum"}, \code{"mean"}, \code{"min"}, \code{"max"}), or a formula calling an
external function with first argument \code{.x} (e.g., \code{~fmedian(.x, na.rm = TRUE)}).}

\item{drop}{Logical. Should unused factor levels be dropped? Default is \code{TRUE}.}

\item{sort}{Logical. If \code{TRUE}, sort the result so the largest groups are shown
on top. Default is \code{FALSE}.}
}
\value{
A data frame of the same type as \code{.data} in the pivoted format.
\code{pivot_longer_()} returns a data frame with more rows and fewer columns.
\code{pivot_wider_()} returns a data frame with fewer rows and more columns.
}
\description{
Functions for pivoting data between long and wide formats.

These are SciViews::R versions of tidyverse functions with standard
evaluation and formula-based non-standard evaluation (ending with underscore
\verb{_}). They work with data.frame, data.table, and tibbles.

\strong{Functions:}
\itemize{
\item \code{pivot_longer_()} - Convert data from wide to long format
\item \code{pivot_wider_()} - Convert data from long to wide format
}
}
\examples{
library(svTidy)

# Create sample wide data
wide_data <- data.frame(
  id = 1:3,
  year = c(2020, 2021, 2022),
  q1 = c(100, 110, 120),
  q2 = c(105, 115, 125),
  q3 = c(110, 120, 130),
  q4 = c(115, 125, 135)
)

# Pivot from wide to long format
wide_data |>
  pivot_longer_(~q1:q4, names_to = "quarter", values_to = "sales")

# Use tidy-select helpers
wide_data |>
  pivot_longer_(~starts_with("q"), names_to = "quarter", values_to = "sales")

# Remove prefix from column names
wide_data |>
  pivot_longer_(
    ~q1:q4,
    names_to = "quarter",
    values_to = "sales",
    names_prefix = "q"
  )

# Control row ordering with cols_vary
wide_data |>
  pivot_longer_(~q1:q4, cols_vary = "slowest")

# Drop NA values
wide_na <- wide_data
wide_na$q3[2] <- NA
wide_na |>
  pivot_longer_(~q1:q4, values_drop_na = TRUE)

# Convert to factors
wide_data |>
  pivot_longer_(~q1:q4, factor = TRUE)

# Create sample long data
long_data <- data.frame(
  id = rep(1:3, each = 4),
  year = rep(c(2020, 2021, 2022), each = 4),
  quarter = rep(c("q1", "q2", "q3", "q4"), 3),
  sales = c(100, 105, 110, 115, 110, 115, 120, 125, 120, 125, 130, 135)
)

# Pivot from long to wide format
long_data |>
  pivot_wider_(names_from = "quarter", values_from = "sales")

# Specify id columns explicitly
long_data |>
  pivot_wider_(
    id_cols = ~c(id, year),
    names_from = "quarter",
    values_from = "sales"
  )

# Add prefix to new column names
long_data |>
  pivot_wider_(
    names_from = "quarter",
    values_from = "sales",
    names_prefix = "sales_"
  )

# Fill missing values
long_data |>
  pivot_wider_(
    names_from = "quarter",
    values_from = "sales",
    values_fill = 0
  )

# Handle multiple values with aggregation
long_dup <- rbind(long_data, long_data[1:3, ])
long_dup |>
  pivot_wider_(
    names_from = "quarter",
    values_from = "sales",
    values_fn = "mean"
  )

# Use custom aggregation function
long_dup |>
  pivot_wider_(
    names_from = "quarter",
    values_from = "sales",
    values_fn = ~sum(.x)
  )

# Multiple names_from and values_from
long_multi <- data.frame(
  id = rep(1:2, each = 4),
  metric = rep(c("sales", "profit"), 4),
  quarter = rep(c("q1", "q2"), each = 2, times = 2),
  value = c(100, 20, 105, 22, 110, 24, 115, 26)
)
long_multi |>
  pivot_wider_(
    names_from = c("quarter", "metric"),
    values_from = "value"
  )

# Round-trip: wide -> long -> wide
wide_data |>
  pivot_longer_(~q1:q4, names_to = "quarter", values_to = "sales") |>
  pivot_wider_(names_from = "quarter", values_from = "sales")

}
\seealso{
\code{\link[tidyr:pivot_longer]{tidyr::pivot_longer()}}, \code{\link[tidyr:pivot_wider]{tidyr::pivot_wider()}}, \code{\link[collapse:pivot]{collapse::pivot()}}
}
