% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/selecting.R
\name{selecting}
\alias{selecting}
\alias{select_}
\alias{pull_}
\alias{rename_}
\alias{rename_with_}
\alias{all_of}
\title{Selecting Functions}
\usage{
select_(.data = (.), ...)

pull_(.data = (.), var = -1, name = NULL, ...)

rename_(.data = (.), ...)

rename_with_(.data = (.), .fn, .cols = ~everything(), ...)

all_of(x)
}
\arguments{
\item{.data}{A data frame (data.frame, data.table, or tibble)}

\item{...}{For \code{select_()} and \code{rename_()}: column names, positions, or
tidy-select expressions. Use formulas for non-standard evaluation
(e.g., \code{~starts_with("x")}). For \code{rename_()}, provide pairs as
\code{new_name = old_name} or \code{new_name = ~old_name}. For \code{rename_with_()}:
additional arguments passed to \code{.fn}.}

\item{var}{For \code{pull_()}: the column to extract. Can be a column name
(as character), a formula with column name on RHS (e.g., \code{~mpg}),
a positive integer for position from left, or a negative integer for
position from right. Default is \code{-1} (last column).}

\item{name}{For \code{pull_()}: optional column to use as names for the resulting
vector. Specified the same way as \code{var}.}

\item{.fn}{For \code{rename_with_()}: a function to apply to column names, or a
formula expression using \code{.x} as the column names (e.g., \code{~toupper(.x)}).}

\item{.cols}{For \code{rename_with_()}: columns to rename. Use tidy-select syntax
with formulas. Default is \code{~everything()} (all columns).}

\item{x}{For \code{all_of()}: a character vector of column names. All must exist
or an error is raised.}
}
\value{
\itemize{
\item \code{select_()} returns a data frame with only the selected columns
\item \code{pull_()} returns a vector (named or unnamed depending on \code{name} parameter)
\item \code{rename_()} returns the data frame with renamed columns
\item \code{rename_with_()} returns the data frame with renamed columns
\item \code{all_of()} returns the input vector (used inside select/rename functions)
}
}
\description{
Functions for selecting, renaming, and extracting columns from a data frame.

These are SciViews::R versions of tidyverse functions with standard
evaluation and formula-based non-standard evaluation (ending with underscore
\verb{_}). They work with data.frame, data.table, and tibbles.

\strong{Functions:}
\itemize{
\item \code{select_()} - Select columns by name, position, or using tidy-select
helpers
\item \code{pull_()} - Extract a single column as a vector
\item \code{rename_()} - Rename columns using new_name = old_name pairs
\item \code{rename_with_()} - Rename columns using a function
\item \code{all_of()} - Helper for selecting all specified columns (errors if missing)
}
}
\examples{
library(svTidy)
data(mtcars)

# Select specific columns by name
mtcars |> select_(~mpg, ~cyl, ~hp)

# Select columns by position
mtcars |> select_(1, 3, 5)

# Select range of columns
mtcars |> select_(~mpg:hp)

# Use tidy-select helpers
mtcars |> select_(~starts_with("d"))
mtcars |> select_(~ends_with("p"))
mtcars |> select_(~contains("a"))

# Exclude columns with minus
mtcars |> select_(~-c(mpg, cyl))

# Select all numeric columns
mtcars |> select_(~where(is.numeric))

# Combine multiple selection methods
mtcars |> select_(~mpg, ~starts_with("d"), ~hp)

# Use all_of() for programmatic selection
cols <- c("mpg", "cyl", "hp")
mtcars |> select_(~all_of(cols))

# Pull a column as a vector (by name)
mtcars |> pull_(~mpg)

# Pull by position (last column)
mtcars |> pull_(-1)

# Pull first column
mtcars |> pull_(1)

# Pull with names from another column
mtcars |> pull_(~mpg, name = ~cyl)

# Rename columns with new_name = old_name
mtcars |> rename_(miles_per_gallon = ~mpg, cylinders = ~cyl)

# Rename using column positions
mtcars |> rename_(miles_per_gallon = 1, cylinders = 2)

# Rename multiple columns
mtcars |>
  rename_(
    miles_per_gallon = ~mpg,
    cylinders = ~cyl,
    horsepower = ~hp
  )

# Rename all columns with a function
mtcars |> rename_with_(toupper)

# Rename using a formula with .x
mtcars |> rename_with_(~paste0("var_", .x))

# Rename with string manipulation
mtcars |> rename_with_(~tolower(.x))
mtcars |> rename_with_(~gsub("_", ".", .x))

# Rename only selected columns
mtcars |> rename_with_(toupper, .cols = ~starts_with("d"))

# Rename specific columns by name
mtcars |> rename_with_(toupper, .cols = c("mpg", "cyl", "hp"))

# Chain operations
mtcars |>
  select_(~mpg, ~cyl, ~hp, ~wt) |>
  rename_(efficiency = ~mpg, weight = ~wt) |>
  arrange_(~cyl)

# Use in data pipeline
mtcars |>
  select_(~where(is.numeric)) |>
  rename_with_(tolower) |>
  filter_(~cyl > 4) |>
  pull_(~mpg)

}
\seealso{
\code{\link[dplyr:select]{dplyr::select()}}, \code{\link[dplyr:pull]{dplyr::pull()}}, \code{\link[dplyr:rename]{dplyr::rename()}},
\code{\link[dplyr:rename]{dplyr::rename_with()}}, \code{\link[dplyr:reexports]{dplyr::all_of()}}, \code{\link[tidyselect:starts_with]{tidyselect::starts_with()}},
\code{\link[tidyselect:starts_with]{tidyselect::ends_with()}}, \code{\link[tidyselect:starts_with]{tidyselect::contains()}}, \code{\link[tidyselect:starts_with]{tidyselect::matches()}},
\code{\link[tidyselect:everything]{tidyselect::everything()}}, \code{\link[collapse:select_replace_vars]{collapse::fselect()}}
}
