% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/summarising.R
\name{summarising}
\alias{summarising}
\alias{summarise_}
\alias{summarize_}
\alias{reframe_}
\alias{count_}
\alias{tally_}
\alias{add_count_}
\alias{add_tally_}
\title{Summarising and Counting Functions}
\usage{
summarise_(
  .data = (.),
  ...,
  .by = NULL,
  .groups = "drop_last",
  .keep.group_vars = TRUE,
  .cols = NULL
)

summarize_(
  .data = (.),
  ...,
  .by = NULL,
  .groups = "drop_last",
  .keep.group_vars = TRUE,
  .cols = NULL
)

reframe_(
  .data,
  ...,
  .by = NULL,
  .groups = "drop",
  .keep.group_vars = TRUE,
  .cols = NULL
)

count_(
  .data = (.),
  ...,
  wt = NULL,
  name = "n",
  sort = FALSE,
  decreasing = TRUE,
  .drop = TRUE,
  add = FALSE
)

tally_(.data = (.), wt = NULL, name = "n", sort = FALSE, decreasing = TRUE)

add_count_(
  .data = (.),
  ...,
  wt = NULL,
  name = "n",
  sort = FALSE,
  decreasing = TRUE,
  .drop = TRUE
)

add_tally_(.data = (.), wt = NULL, name = "n", sort = FALSE, decreasing = TRUE)
}
\arguments{
\item{.data}{A data frame (data.frame, data.table, or tibble)}

\item{...}{For \code{summarise_()} and \code{reframe_()}: name-value pairs of summary
functions. Names are column names in the output; values are expressions to
compute. Use formulas for non-standard evaluation (e.g., \code{~mean_mpg = mean(~mpg)}).
For \code{count_()} and \code{add_count_()}: grouping variables specified as formulas
(e.g., \code{~cyl}, \code{~gear}) or character names. Can include named expressions
to create new grouping variables before counting.}

\item{.by}{Optional temporary grouping variables for per-group computations.
Provide as formulas (e.g., \code{~group_col}) or character names. Groups are
temporary and not preserved in the output. Cannot be used with grouped
data frames.}

\item{.groups}{Control grouping of the result. Options:
\itemize{
\item \code{"drop_last"} (default) - Drop the last grouping level
\item \code{"drop"} - Remove all grouping
\item \code{"keep"} - Keep all grouping levels
\item \code{"rowwise"} - Not implemented
For \code{reframe_()}, only \code{"drop"} is allowed.
}}

\item{.keep.group_vars}{Logical. If \code{TRUE} (default), keep grouping variables
in the result.}

\item{.cols}{Optional character vector of column names to operate on. Currently
only \code{NULL} (default) is implemented.}

\item{wt}{For \code{count_()}, \code{tally_()}, \code{add_count_()}, and \code{add_tally_()}:
frequency weights. Can be \code{NULL} (default, counts rows), a numeric vector,
a column name as character, or a formula (e.g., \code{~weight_col}).}

\item{name}{Character string specifying the name of the count column created
in the output. Default is \code{"n"}.}

\item{sort}{Logical. If \code{TRUE}, sort the result by the count column in
decreasing order (or as specified by \code{decreasing}). Default is \code{FALSE}.}

\item{decreasing}{Logical. If \code{TRUE} (default), sort counts in decreasing
order when \code{sort = TRUE}.}

\item{.drop}{Logical. If \code{TRUE} (default), drop unused factor levels. Note:
\code{.drop = FALSE} is not yet implemented in \code{count_()}.}

\item{add}{Logical. If \code{TRUE}, add the count column to the original data
frame instead of returning a summary. Default is \code{FALSE}.}
}
\value{
\itemize{
\item \code{summarise_()} returns a data frame with one row per group (or one row if
ungrouped), containing the summary statistics. Grouping depends on \code{.groups}.
\item \code{reframe_()} returns an ungrouped data frame (can have any number of rows
per group).
\item \code{count_()} returns a data frame with one row per unique combination of
grouping variables, plus a count column.
\item \code{tally_()} returns a data frame with one row per group showing the count.
\item \code{add_count_()} returns the original data with an additional count column.
\item \code{add_tally_()} returns the original data with an additional count column.
}
}
\description{
Functions for summarising data and counting observations in data frames.

These are SciViews::R versions of tidyverse functions with standard
evaluation and formula-based non-standard evaluation (ending with underscore
\verb{_}). They work with data.frame, data.table, and tibbles.

\strong{Functions:}
\itemize{
\item \code{summarise_()} / \code{summarize_()} - Compute summary statistics for groups
\item \code{reframe_()} - Similar to summarise but always returns ungrouped data
\item \code{count_()} - Count observations by group
\item \code{tally_()} - Count total observations (wrapper around count_)
\item \code{add_count_()} - Add count column to data frame
\item \code{add_tally_()} - Add total count column to data frame
}
}
\note{
The \code{summarise_()} function does not support \code{n()} as does \code{\link[dplyr:summarise]{dplyr::summarise()}}.
You can use \code{\link[svBase:fstat_functions]{svBase::fn()}} instead, but then you must give a variable name as
argument. The \code{\link[svBase:fstat_functions]{svBase::fn()}} alternative can also be used in \code{\link[dplyr:summarise]{dplyr::summarise()}}
for homogeneous syntax between the two.
}
\examples{
library(svTidy)
data(mtcars)

# Basic summarise - single summary statistic
mtcars |> summarise_(mean_mpg = ~mean(mpg))

# Multiple summary statistics
mtcars |>
  summarise_(
    mean_mpg = ~mean(mpg),
    sd_mpg   = ~sd(mpg),
    max_hp   = ~max(hp)
  )

# Summarise by groups
mtcars |>
  group_by_(~cyl) |>
  summarise_(
    mean_mpg = ~mean(mpg),
    mean_hp  = ~mean(hp)
  )

# Use .by for temporary grouping
mtcars |>
  summarise_(
    mean_mpg = ~mean(mpg),
    count    = ~length(mpg),
    .by = 'cyl'
  )

# Multiple grouping variables with .by
mtcars |>
  summarise_(
    mean_mpg = ~mean(mpg),
    .by = c('cyl', 'gear')
  )

# Control grouping of result
mtcars |>
  group_by_(~cyl, ~gear) |>
  summarise_(mean_mpg = ~mean(mpg), .groups = "drop")

mtcars |>
  group_by_(~cyl, ~gear) |>
  summarise_(mean_mpg = ~mean(mpg), .groups = "keep")

# Using standard evaluation (ungrouped data only)
mtcars |> summarise_(mean_mpg = mean(mtcars$mpg))

# reframe_() for summaries returning multiple rows per group
mtcars |>
  group_by_(~cyl) |>
  reframe_(quantile_mpg = ~quantile(mpg, c(0.25, 0.5, 0.75)))

# Count observations by group
mtcars |> count_(~cyl)

# Count by multiple variables
mtcars |> count_(~cyl, ~gear)

# Count with sorting
mtcars |> count_(~cyl, sort = TRUE)

# Count in increasing order
mtcars |> count_(~cyl, sort = TRUE, decreasing = FALSE)

# Count with weights
mtcars |> count_(~cyl, wt = ~mpg)

# Count with computed grouping variable
mtcars |> count_(high_mpg = ~mpg > 20)

# Combine grouping and computation
mtcars |> count_(~cyl, high_hp = ~hp > 150)

# tally_() - count rows (optionally by existing groups)
mtcars |> tally_()

mtcars |>
  group_by_(~cyl) |>
  tally_()

# tally with weights
mtcars |>
  group_by_(~cyl) |>
  tally_(wt = ~hp)

# add_count_() - add count column without collapsing
mtcars |> add_count_(~cyl)

# add_count with custom column name
mtcars |> add_count_(~cyl, name = "n_cyl")

# add_count by multiple variables
mtcars |> add_count_(~cyl, ~gear)

# add_tally_() - add total count to each row
mtcars |> add_tally_()

mtcars |>
  group_by_(~cyl) |>
  add_tally_()

# Chain operations
mtcars |>
  count_(~cyl, ~gear, sort = TRUE) |>
  mutate_(pct = ~n/sum(n) * 100)

# Use with filtering
mtcars |>
  add_count_(~cyl) |>
  filter_(~n > 10)

}
\seealso{
\code{\link[dplyr:summarise]{dplyr::summarise()}}, \code{\link[dplyr:reframe]{dplyr::reframe()}}, \code{\link[dplyr:count]{dplyr::count()}}, \code{\link[dplyr:count]{dplyr::tally()}},
\code{\link[dplyr:count]{dplyr::add_count()}}, \code{\link[dplyr:count]{dplyr::add_tally()}}, \code{\link[collapse:fsummarise]{collapse::fsummarise()}},
\code{\link[collapse:fcount]{collapse::fcount()}}, \code{\link[svBase:fstat_functions]{svBase::fn()}}
}
